--[[---------------------------------------------------------------------------
	Chocolatier Two Simulator: Factories
	Copyright (c) 2006-2007 Big Splash Games, LLC. All Rights Reserved.
--]]---------------------------------------------------------------------------

local defaultProduct = "b_01"

require("sim/building.lua")

-- Factory class definition, derived from LBuilding
LFactory = LBuilding:subclass
{
	__tostring = function(t) return "{Factory:" .. tostring(t.name) .. "}" end,
	_Factories = {},
	_ByPort = {},
	
	globalNeeds = {},
	ingredientStatus = {},
	weeksPerColumn = { 4,4,8,8,8,10,10 },
}

-------------------------------------------------------------------------------
-- Functions for data description

function Factory(t) return LFactory:new(t) end

-------------------------------------------------------------------------------
-- "constructor"

function LFactory:new(t)
	t = LBuilding.new(self, t)

	-- Default configuration will be provided on ResetOne
	self.config = {}
	self.machines = { square=true }
	self.needs = {}

	self.visited = nil
	
	table.insert(self._Factories, t)
	return t
end

-------------------------------------------------------------------------------
-- Convenient data access

function LFactory:AllFactories()
	return bsgArrayIterator(self._Factories)
end

function LFactory:ByPort(name)
	return self._ByPort[name]
end

function LFactory:SetPort(p)
	LBuilding.SetPort(self, p)
	self.port = p
	self._ByPort[p.name] = self
end

-------------------------------------------------------------------------------
-- Popups

function LFactory:InfoRolloverTarget()
	return "(LFactory:ByName('"..self.name.."')):FactoryRollover()"
end

function LFactory:FactoryRollover()
	if self.owned then return self:OwnedFactoryRollover()
	else return self:NonOwnedFactoryRollover()
	end
end

function LFactory:NonOwnedFactoryRollover()
	local h = bsgFontHeight(popupFont)
	local label = "#"..GetString("factory_forsale", bsgDollars(self.price))
	local r =
	{
		x=0,y=0, color=PopupColor, inset=2, alpha=.8,
		AppendStyle { font = popupFont, flags=kHAlignLeft+kVAlignTop },
		TightText { x=0,y=0, label=label },
	}
	return MakeRollover(r)
end

function LFactory:OwnedFactoryRollover()
	local h = bsgFontHeight(popupFont)
	
	local name = LabelString(self.port.name)
	if self.product then name = name .. " - " .. GetString(self.product.name) end
	local r =
	{
		x=0,y=0, color=PopupColor, inset=2, alpha=.8,
		AppendStyle { font = popupFont, flags=kHAlignLeft+kVAlignTop },
		TightText { x=0,y=0, label=name },
	}
	
	weeksPerColumn = LFactory.weeksPerColumn[gSim.rank + 1]
	
	local y = h
	if self.rate and self.rate > 0 then
		local rate = "#"..GetString("factory_perweek", tostring(self.rate))
		table.insert(r, TightText { x=0,y=y, label=rate })
		y = y + h
		
		if self.weeks > 0 then
			local w = nil
			if self.weeks < 2 then w = "factory_short"
			else w = "#"..GetString("factory_weeks", tostring(self.weeks))
			end
			table.insert(r, TightText { x=0, y=y, label=w })
			y = y + h
		end
	else
		table.insert(r, TightText { x=0, y=y, label="factory_zero" })
		y = y + h
	end
	
	if self.weeks < weeksPerColumn then
		local scale = h/ui.itemHeight
		local x = 5 + bsgFloor(ui.itemWidth * scale + 1)
		for ing,count in pairs(self.product.recipe) do
			local item = LItem:ByName(ing)
			local needs = LFactory.globalNeeds[ing] or count
			local weeks = item.inventory / needs
			local percentage = weeks / weeksPerColumn
--			if weeks < weeksPerColumn then

			if percentage < 1 then
				local label = GetString(ing)
				if weeks == 0 then label = WorsePriceColor .. GetString("factory_need", label)
				else label = GetString("factory_low", label)
				end

				if item.lastport then
					label = label.." - "..GetString("last_seen_port", GetString(item.lastport), bsgDollars(item.lastprice))
				end
				
				table.insert(r, Bitmap {x=5,y=y, image="item/"..ing, scale=h/ui.itemHeight })
				local s = "#"..GetString(ing)
				table.insert(r, TightText {x=x,y=y, label=label })
				y = y + h
			end
		end
	end

	return MakeRollover(r)
end

-------------------------------------------------------------------------------
-- Reset

function LFactory:ResetOne()
	LBuilding.ResetOne(self)

	self.config = {}
	self.machines = { square=true }
	self.needs = {}
	self:SetConfiguration(defaultProduct, 0)
	self.visited = nil
end

-------------------------------------------------------------------------------
-- Ownership

function LFactory:MarkOwned()
	gSim.factoriesOwned = gSim.factoriesOwned or 0
	gSim.factoriesOwned = gSim.factoriesOwned + 1
	LBuilding.MarkOwned(self)
	if self.port then
		self.port.factoryOwned = true
	end
end

-------------------------------------------------------------------------------
-- Load and Save

function LFactory:SaveOne()
	local t = LBuilding.SaveOne(self)
	if self.owned then
		assert(t)
		t.machines = self.machines
		t.config = self.config
		t.product = self.product.name
	else
		assert(not t)
		if self.visited then
			t = { visited=true }
		end
	end
	return t
end

function LFactory:LoadOne(t)
	LBuilding.LoadOne(self, t)
	if self.owned then
		self.machines = t.machines
		self.config = t.config
		self:SetConfiguration(t.product)
	else
		self.visited = t.visited
	end
end

-------------------------------------------------------------------------------
-- Building interaction

function LFactory:SelectCharacterFreePlay() return self.character[1] end

function LFactory:OnActivate()
	gActiveBuilding = self
	self.visited = true
	local char,quest = self:SelectCharacter()
	
	if self.owned then
		gCurrentFactory = self
		DisplayDialog { "ui/factory_status.lua", factory=self }
	else
		local price = self.price or 10000
	
		if price > gSim.money then
			DisplayDialog { "ui/chartalk.lua", char=char, body="#"..GetString("factory_expensive", bsgDollars(price)), ok="ok" }
		else
			local yn = DisplayDialog { "ui/chartalk.lua", char=char, body="#"..GetString("factory_purchase", bsgDollars(price)), yes="yes",no="no" }
			if yn == "yes" then
				gSim:AdjustMoney(-price)
				self:MarkOwned()
				local msg = GetString("factory_purchased", GetString(self.name), bsgDollars(price))
				SetLedgerContents("factories")
				gSim:QueueMessage(msg)		-- No need to flush message, ledger guaranteed on factories
				DisplayDialog { "ui/factory_status.lua", factory=self }
			end
		end
	end
	self:ExitBuilding()
	gCurrentFactory = nil
	gDialogTable.factory = nil
end

-------------------------------------------------------------------------------
-- Factory configuration and production

function LFactory:SetConfiguration(productName, rate)
	self.product = LItem:ByName(productName)
	if not self.product then
		productName = defaultProduct
		self.product = LItem:ByName(productName)
	end

	rate = rate or self.config[productName]
	rate = rate or 0
	self.config[productName] = rate
	self.rate = rate
	
	-- Gather needs for this factory
	self.needs = {}
	if self.rate > 0 then
		for ing,count in pairs(self.product.recipe) do
			self.needs[ing] = count * self.rate
		end
	end
	
	-- Update needs for all factories combined
	LFactory.globalNeeds = {}
	for f in LFactory:AllFactories() do
		for ing,count in pairs(f.needs) do
			LFactory.globalNeeds[ing] = (LFactory.globalNeeds[ing] or 0) + count
		end
	end
end

function LFactory:GatherIngredientStatus()
	-- Gather global ingredient status into a coherent structure for ordered display
	LFactory.ingredientStatus = {}
	for ing in LItem:AllIngredients() do
		if LFactory.globalNeeds[ing.name] then
			local weeks = ing.inventory / LFactory.globalNeeds[ing.name]
			table.insert(LFactory.ingredientStatus, { name=ing.name, ing=ing, need=LFactory.globalNeeds[ing.name], have=ing.inventory, weeks=weeks })
		end
	end
end

function LFactory:ProjectProduction()
	-- Figure out how many weeks of each ingredient we have
	local weeks = {}
	for ing,count in pairs(LFactory.globalNeeds) do
		local item = LItem:ByName(ing)
		if count > 0 then
			weeks[ing] = bsgFloor(item.inventory / count)
		end
	end
	
	local stall = false
	
	-- Figure out usage at each factory
	for f in LFactory:AllFactories() do
		if f.owned and f.rate > 0 then
			f.shortage = {}
			f.weeks = 999999
			local idle = f.idle
			f.idle = false
			for ing,count in pairs(f.product.recipe) do
				if weeks[ing] < f.weeks then f.weeks = weeks[ing] end
				
				local item = LItem:ByName(ing)
				if item.inventory < f.needs[ing] then f.shortage[ing] = true end
				if item.inventory < count then
					if not idle then stall = true end
					f.weeks = 0
					f.idle = true
				end
			end
		else
			f.weeks = 0
			f.idle = true
		end
	end
	
	-- If this is the first week of a stall, play the warning sound
	if stall and not gSim.stall then PlaySound("factory_stall") end
	
	-- Gather ingredient info for display
	self:GatherIngredientStatus()
end

function LFactory:DoProduction()
	-- Determine how many products we could create from the current pool of ingredients, up
	-- to the factory configuration
	local production = self.rate
	for ing,count in pairs(self.product.recipe) do
		local item = LItem:ByName(ing)
		local cando = bsgFloor(item.inventory / count)
		if cando < production then production = cando end
	end
	
	-- Do the production, adjust inventory
	if production > 0 then
		for ing,count in pairs(self.product.recipe) do
			local item = LItem:ByName(ing)
			local used = count * production
			item.inventory = item.inventory - used
			if used > 0 then item.usetime = gSim.weeks end
		end
		
		self.product.inventory = self.product.inventory + production
		self.product.usetime = gSim.weeks
	else
		self.idle = true
	end
	
	return production
end

function LFactory:ProductionRun()
	local change = false
	gSim.stall = false
	for f in LFactory:AllFactories() do
		if f.owned and f.rate > 0 then
			if f:DoProduction() == 0 then gSim.stall = true
			else change = true
			end
		end
	end
	
	if gSim.stall then
		gSim.stallweeks = gSim.stallweeks + 1
		
		-- This is part of a Tick, messages will be flushed at end of Tick
		gSim:QueueMessage(GetString("factory_stop"))
	end
	
	if change then
		gSim:InventoryChanged()
		self:GatherIngredientStatus()
	end
end
